// the semi-colon before function invocation is a safety net against concatenated
// scripts and/or other plugins which may not be closed properly.
;(function ($, window, document, undefined) {

    "use strict";

    // Create the defaults once
    var pluginName = "YatraCalendar",
        defaults = {
            months: ['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'], //string of months starting from january
            days: ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'],
            displayYear: true, // display year in header
            fixedStartDay: true, // Week begin always by monday or by day set by number 0 = sunday, 7 = saturday, false = month always begin by first day of the month
            displayEvent: true, // display existing event
            disableEventDetails: false, // disable showing event details
            disableEmptyDetails: false, // disable showing empty date details
            events: [], // List of event
            onInit: function (calendar) {
            }, // Callback after first initialization
            onCalendarLoad: function () {

            },
            onMonthChange: function (month, year) {
            }, // Callback on month change,
            onBeforeMonthChange: function (month, year) {
            }, // Callback on month change
            onDateSelect: function (date, el) {
            }, // Callback on date selection
            onEventSelect: function () {
            },              // Callback fired when an event is selected     - see $(this).data('event')
            onEventCreate: function ($el) {
            },          // Callback fired when an HTML event is created - see $(this).data('event')
            onDayCreate: function ($el, fd, d, m, y) {
            }  // Callback fired when an HTML day is created   - see $(this).data('today'), .data('todayEvents')
        };

    // The actual plugin constructor
    function Plugin(element, options) {
        this.element = element;
        this.settings = $.extend({}, defaults, options);
        this._defaults = defaults;
        this._name = pluginName;
        this.currentDate = new Date();
        this.init();
    }

    // Avoid Plugin.prototype conflicts
    $.extend(Plugin.prototype, {
        init: function () {
            var container = $(this.element);
            var todayDate = this.currentDate;

            var calendar = $('<div class="yatra-calendar"></div>');
            var header = $('<header>' +
                '<a class="yatra-calendar-btn btn-prev" href="#"><span class="fas fa-chevron-left"></span></a>' +
                '<h2 class="month"></h2>' +
                '<a class="yatra-calendar-btn btn-next" href="#"><span class="fas fa-chevron-right"></span></a>' +
                '</header>');

            this.updateHeader(todayDate, header);
            calendar.append(header);

            this.buildCalendar(todayDate, calendar);

            container.append(calendar);

            this.bindEvents();

            this.settings.onInit(this);
        },

        //Update the current month header
        updateHeader: function (date, header) {
            var monthText = this.settings.months[date.getMonth()];
            monthText += this.settings.displayYear ? ' <div class="year">' + date.getFullYear() : '</div>';
            header.find('.month').html(monthText);
        },

        //Build calendar of a month from date
        buildCalendar: function (fromDate, calendar) {

            var plugin = this;

            calendar.find('table').remove();

            var body = $('<table></table>');
            var thead = $('<thead></thead>');
            var tbody = $('<tbody></tbody>');

            //setting current year and month
            var y = fromDate.getFullYear(), m = fromDate.getMonth();

            //first day of the month
            var firstDay = new Date(y, m, 1);
            //last day of the month
            var lastDay = new Date(y, m + 1, 0);
            // Start day of weeks
            var startDayOfWeek = firstDay.getDay();

            if (this.settings.fixedStartDay !== false) {
                // Backward compatibility
                startDayOfWeek = this.settings.fixedStartDay === true ? 1 : this.settings.fixedStartDay;

                // If first day of month is different of startDayOfWeek
                while (firstDay.getDay() !== startDayOfWeek) {
                    firstDay.setDate(firstDay.getDate() - 1);
                }
                // If last day of month is different of startDayOfWeek + 7
                while (lastDay.getDay() !== ((startDayOfWeek + 6) % 7)) {
                    lastDay.setDate(lastDay.getDate() + 1);
                }
            }
            //Header day in a week ( (x to x + 7) % 7 to start the week by monday if x = 1)
            for (var i = startDayOfWeek; i < startDayOfWeek + 7; i++) {
                thead.append($('<td class="yatra-tippy-tooltip" data-tippy-content="' + this.settings.days[i] + '">' + this.settings.days[i % 7].substring(0, 3) + '</td>'));
            }

            //For firstDay to lastDay
            for (var day = firstDay; day <= lastDay; day.setDate(day.getDate())) {
                var tr = $('<tr></tr>');
                //For each row
                for (var i = 0; i < 7; i++) {
                    var td = $('<td><div class="day" data-date="' + day.toISOString() + '">' + day.getDate() + '</div></td>');

                    var $day = td.find('.day');

                    //if today is this day
                    if (day.toDateString() === (new Date).toDateString()) {
                        $day.addClass("today");
                    }

                    //if day is not in this month
                    if (day.getMonth() != fromDate.getMonth()) {
                        $day.addClass("wrong-month disabled");
                    }


                    // filter today's events
                    var hasEnableDay = plugin.hasEnableDay(day);
                    var hasDisableDay = plugin.hasDisableDay(day);


                    if (!hasEnableDay || hasDisableDay) {
                        $day.addClass('disabled');
                    }

                    // simplify further customization
                    this.settings.onDayCreate($day, this.formatDate(day), day.getDate(), m, y,);

                    tr.append(td);
                    day.setDate(day.getDate() + 1);
                }
                tbody.append(tr);
            }

            body.append(thead);
            body.append(tbody);
            calendar.append(body);
            this.settings.onCalendarLoad();
        },
        changeMonth: function (value) {
            this.currentDate.setMonth(this.currentDate.getMonth() + value, 1);
            this.settings.onBeforeMonthChange(this.currentDate.getMonth(), this.currentDate.getFullYear())
            this.buildCalendar(this.currentDate, $(this.element).find('.yatra-calendar'));
            this.updateHeader(this.currentDate, $(this.element).find('.yatra-calendar header'));
            this.settings.onMonthChange(this.currentDate.getMonth(), this.currentDate.getFullYear())
        },
        //Init global events listeners
        bindEvents: function () {
            var plugin = this;

            //Click previous month
            $(plugin.element).on('click', '.btn-prev', function (e) {
                plugin.changeMonth(-1)
                e.preventDefault();
            });

            //Click next month
            $(plugin.element).on('click', '.btn-next', function (e) {
                plugin.changeMonth(1);
                e.preventDefault();
            });

            //Binding day event
            $(plugin.element).on('click', '.day:not(.disabled)', function (e) {
                var date = new Date($(this).data('date'));
                $(this).closest('table').find('td').removeClass('active');
                $(this).closest('td').addClass('active');
                plugin.settings.onDateSelect(plugin.formatDate(date), $(this));
            });


        },
        addEvent: function (newEvent) {
            var plugin = this;
            // add the new event to events list
            plugin.settings.events = [...plugin.settings.events, newEvent]
            this.buildCalendar(this.currentDate, $(this.element).find('.yatra-calendar'));
        },

        formatDate: function (date) {
            var d = new Date(date),
                month = '' + (d.getMonth() + 1),
                day = '' + d.getDate(),
                year = d.getFullYear();

            if (month.length < 2)
                month = '0' + month;
            if (day.length < 2)
                day = '0' + day;

            return [year, month, day].join('-');
        },

        hasEnableDay: function (d) {

            var plugin = this;

            if (typeof plugin.settings.enable === undefined) {

                return true;
            }

            var enable = [];

            if (typeof plugin.settings.enable === "function") {

                enable = plugin.settings.enable(d);
            } else if (typeof plugin.settings.enable === "array") {
                enable = plugin.settings.enable;
            }
            if ($.inArray(this.formatDate(d), enable) !== -1) {

                return true;
            }
            return false;


        },

        hasDisableDay: function (d) {

            var plugin = this;

            if (typeof plugin.settings.disable === undefined) {

                return false;
            }
            if ($.inArray(this.formatDate(d), plugin.settings.disable) !== -1) {

                return true;
            }
            return false;


        },
        isDayBetween: function (d, dStart, dEnd) {
            dStart.setHours(0, 0, 0);
            dEnd.setHours(23, 59, 59, 999);
            d.setHours(12, 0, 0);

            return dStart <= d && d <= dEnd;
        },
        formatDateEvent: function (dateStart, dateEnd) {
            var formatted = '';
            formatted += this.settings.days[dateStart.getDay()] + ' - ' + dateStart.getDate() + ' ' + this.settings.months[dateStart.getMonth()].substring(0, 3);

            if (dateEnd.getDate() !== dateStart.getDate()) {
                formatted += ' to ' + dateEnd.getDate() + ' ' + this.settings.months[dateEnd.getMonth()].substring(0, 3)
            }
            return formatted;
        }
    });

    // A really lightweight plugin wrapper around the constructor,
    // preventing against multiple instantiations
    $.fn[pluginName] = function (options) {
        return this.each(function () {
            if (!$.data(this, "plugin_" + pluginName)) {
                $.data(this, "plugin_" + pluginName, new Plugin(this, options));
            }
        });
    };
})(jQuery, window, document);
