<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Tourm Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class Tourm_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );


        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);


		// Specific Register widget scripts

		// add_action( 'elementor/frontend/after_register_scripts', [ $this, 'tourm_regsiter_widget_scripts' ] );
		// add_action( 'elementor/frontend/before_register_scripts', [ $this, 'tourm_regsiter_widget_scripts' ] );


        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'tourm_elementor_widget_categories' ] );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'tourm' ),
			'<strong>' . esc_html__( 'Tourm Core', 'tourm' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'tourm' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'tourm' ),
			'<strong>' . esc_html__( 'Tourm Core', 'tourm' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'tourm' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'tourm' ),
			'<strong>' . esc_html__( 'Tourm Core', 'tourm' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'tourm' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		$widget_register = \Elementor\Plugin::instance()->widgets_manager;

		// Header Include file & Widget Register
		require_once( TOURAM_ADDONS . '/header/header.php' );
		require_once( TOURAM_ADDONS . '/header/tourm-megamenu.php' );

		$widget_register->register ( new \Tourm_Header() );
		$widget_register->register ( new \Tourm_Megamenu() );


		// Include All Widget Files
		foreach($this->Tourm_Include_File() as $widget_file_name){
			require_once( TOURAM_ADDONS . '/widgets/tourm-'."$widget_file_name".'.php' );
		}
		// All Widget Register
		foreach($this->Tourm_Register_File() as $name){
			$widget_register->register ( $name );
		}
		
	}

	public function Tourm_Include_File(){
		return [
			// 'listing-gallery', 
			// 'listing-features', 
			// 'listing-floorplan', 
			// 'listing-video', 
			'banner', 
			'section-title', 
			'tour-category', 
			'tour-destinations', 
			'animated-shape', 
			'button', 
			'image', 
			'features',
			'gallery-image', 
			'counterup', 
			'team', 
			'testimonial', 
			'brand-logo', 
			'contact-form', 
			'offer-area', 
			'countdown', 
			'faq', 
			'blog', 
			'trip', 
			'price',
			'contact-info', 
			'video', 
			'shape', 
			'menu-select',
			'newsletter', 
			'banners2', 
			'banner-v2', 
			'resort', 
			'cart', 
			'arrows', 
			// 'project',
			// 'team-info', 
			// 'custom-listing', 

		];
	}

	public function Tourm_Register_File(){
		return [
			// new \Tourm_Listing_Gallery(),
			// new \Tourm_Listing_Features(),
			// new \Tourm_Listing_FloorPlan(),
			// new \Tourm_Listing_Video(),
			new \Tourm_Banner() ,
			new \Tourm_Section_Title(),
			new \tourm_Tour_Category(),
			new \tourm_Tour_Destinations(),
			new \Tourm_Animated_Shape(),
			new \Tourm_Button(),
			new \Tourm_Image(),
			new \tourm_Feature(),
			new \tourm_Slider_Image(),
			new \Tourm_Counterup(),
			new \Tourm_Team(),
			new \Tourm_Testimonial(),
			new \Tourm_Brand_Logo(),
			new \Tourm_Contact_Form(),
			new \tourm_Offer(),
			new \Tourm_Offer_Date_Widget(),
			new \Tourm_Faq(),
			// new \Tourm_Blog(),
			new \tourm_Tour_Trip(),
			new \Tourm_Price(),
			new \Tourm_Contact_Info(),
			new \tourm_Video(),
			new \tourm_Shape(),
			new \Tourm_Menu(),
			new \tourm_Newsletter(),
			new \tourm_Blog(),
			new \Tourm_Banner1(),
			new \Tourm_Banner_V2(),
			new \tourm_Resort(),
			new \Tourm_Cart(),
			new \Tourm_Arrows(),
			
			// new \Tourm_Service(),
			// new \tourm_Slider_Image(),
			// new \tourm_Project(),
			// new \Tourm_Team_info(),
			// new \tourm_CustoM_Listing(),

		];
	}

    public function widget_scripts() {

        // wp_enqueue_script(
        //     'tourm-frontend-script',
        //     TOURAM_PLUGDIRURI . 'assets/js/tourm-frontend.js',
        //     array('jquery'),
        //     false,
        //     true
		// );

	}


    function tourm_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'tourm',
            [
                'title' => __( 'Tourm', 'tourm' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'tourm_listing',
            [
                'title' => __( 'Tourm Listing', 'tourm' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'tourm_footer_elements',
            [
                'title' => __( 'Tourm Footer Elements', 'tourm' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'tourm_header_elements',
            [
                'title' => __( 'Tourm Header Elements', 'tourm' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

Tourm_Extension::instance();